//=============================================================================
// SkillMastery.js
// ----------------------------------------------------------------------------
// (C)2025 kuroame koubou
// This software is released under the MIT License.
// http://opensource.org/licenses/mit-license.php
// ----------------------------------------------------------------------------
// Version
// 0.9.0 2025/02/07 初版
// ----------------------------------------------------------------------------
// [Blog]   : https://ci-en.net/creator/14684
// [Twitter]: https://twitter.com/kuroVelter9623
//=============================================================================

/*:
 * @plugindesc スキル使い込み機構プラグイン
 * @target MZ
 * @author くろあめ
 * 
 * @param Personal Count
 * @text 使い込み値個人保有フラグ
 * @desc スキル個別/系統別の使い込み値を、個人で蓄積するかどうかののフラグです。
 * @type select
 * @option 個人蓄積
 * @value 1
 * @option パーティ蓄積
 * @value 2
 * @default 2
 * 
 * @param Skill Mastery Max Count
 * @text スキル個別使い込み最大カウント
 * @desc スキル個別で蓄積する使い込み値のカンスト値です。
 * @default 1000
 * @type number
 * 
 * @param Skill Mastery Max Damage Amplify
 * @text スキル個別使い込み増幅レート
 * @desc スキル個別の使い込み値がこの値のとき、2倍の補正が付きます。
 * @default 10000
 * @type number
 * 
 * @param Lineage Mastery Max Count
 * @text 系統使い込み値最大カウント
 * @desc スキル個別で蓄積する使い込み値のカンスト値です。
 * @default 2000
 * @type number
 * 
 * @param Lineage Mastery Max Damage Amplify
 * @text 系統使い込み値幅レート
 * @desc 系統使い込み値がこの値のとき、2倍の補正が付きます。
 * @default 20000
 * @type number
 *
 * @help
 * スキル個別、および特定のタグを付与されたスキル系列の
 * スキル使用回数を記録し
 * その使用回数に応じて、最終威力に補正をかけます。
 * 
 * スキル個別の使い込み値蓄積、および威力補正は
 * プラグインを導入するだけで動作します
 * カンスト使用回数、および最終補正を設定してください。
 *
 * スキル個別の使い込み値は
 * スキルのみ、蓄積・適用されます
 * アイテムは効果対象外です。
 * 
 * 系統使い込み値は、スキルやアイテムのメモ欄に
 * 特定の系統に属することを示すタグを書き込むことで
 * そのスキル・アイテムを使用するごとに
 * 系統使い込み値が蓄積され、威力に反映されていきます。
 * 
 * スキル・アイテムのメモ欄に
 * <MasteryLineage:n> と記載すると
 * そのスキル・アイテムは、n番系統のスキルと見なされ
 * n番系統の使い込み値の蓄積、および威力反映の対象となります
 * こちらは、スキルの他にアイテムにも指定することが可能です
 * 系統番号は数値で指定してください。
 *
 * 利用形態（商用、18禁利用等）について制限はありません
 * 作者への使用報告や、ReadMeなどへ作者・サイト名の記載も任意となります。
 *
 * ただし、作者はこれらのプラグインについて
 * 基本的に一切のサポートを行いません
 * 何か問題が起きた時、ご自身で解決できる方のみご利用ください。
 */


(function() {

// パラメータ用意
const parameters                     = PluginManager.parameters('SkillMastery');
const PersonalCount                  = Number(parameters['Personal Count']);
const SkillMasteryMaxCount           = Number(parameters['Skill Mastery Max Count']);
const SkillMasteryMaxDamageAmplify   = Number(parameters['Skill Mastery Max Damage Amplify']);
const LineageMasteryMaxCount         = Number(parameters['Lineage Mastery Max Count']);
const LineageMasteryMaxDamageAmplify = Number(parameters['Lineage Mastery Max Damage Amplify']);


// アクターの個別使い込み値出し入れ関数

// 個別使い込み値の初期化・クリア
Game_Actor.prototype.clearSkillMasterySavevalues = function() {
    this._SkillMasterySavevalues = [];
};

// 個別使い込み値の蓄積
Game_Actor.prototype.setSkillMasterySavevalues = function(index) {
    if(!this._SkillMasterySavevalues) this.clearSkillMasterySavevalues();
    if (this._SkillMasterySavevalues[index] === undefined){
        this._SkillMasterySavevalues[index] = 0;
    }
    // 共有フラグのON/OFFに関わらず、蓄積は個人/パーティの両方に行う
    this._SkillMasterySavevalues[index] = Math.min(this._SkillMasterySavevalues[index] + 1 ,SkillMasteryMaxCount);
    $gameParty.setSkillMasterySavevalues(index);
};

// 個別使い込み値の取得 フラグ次第でパーティ情報から読み出す
Game_Actor.prototype.getSkillMasterySavevalues = function(index) {
    if(PersonalCount == 1){
        if(!this._SkillMasterySavevalues) this.clearSkillMasterySavevalues();
        if (this._SkillMasterySavevalues[index] === undefined){
            this._SkillMasterySavevalues[index] = 0;
        }
        return this._SkillMasterySavevalues[index];
    }
    else{
        return $gameParty.getSkillMasterySavevalues(index);
    }
};


// パーティの個別使い込み値出し入れ

// 個別使い込み値の初期化・クリア
Game_Party.prototype.clearSkillMasterySavevalues = function() {
    this._SkillMasterySavevalues = [];
};

// 個別使い込み値の蓄積
Game_Party.prototype.setSkillMasterySavevalues = function(index) {
    if(!this._SkillMasterySavevalues) this.clearSkillMasterySavevalues();
    if (this._SkillMasterySavevalues[index] === undefined){
        this._SkillMasterySavevalues[index] = 0;
    }
    this._SkillMasterySavevalues[index] = Math.min(this._SkillMasterySavevalues[index] + 1 ,SkillMasteryMaxCount);
    
};

// 個別使い込み値の取得
Game_Party.prototype.getSkillMasterySavevalues = function(index) {
    if(!this._SkillMasterySavevalues) this.clearSkillMasterySavevalues();
    if (this._SkillMasterySavevalues[index] === undefined){
        this._SkillMasterySavevalues[index] = 0;
    }
    return this._SkillMasterySavevalues[index];
};


// アクターの系統使い込み値出し入れ関数

// 系統使い込み値の初期化・クリア
Game_Actor.prototype.clearLineageMasterySavevalues = function() {
    this._LineageMasterySavevalues = [];
};

// 系統使い込み値の蓄積
Game_Actor.prototype.setLineageMasterySavevalues = function(index) {
    if(!this._LineageMasterySavevalues) this.clearLineageMasterySavevalues();
    if (this._LineageMasterySavevalues[index] === undefined){
        this._LineageMasterySavevalues[index] = 0;
    }
    // 共有フラグのON/OFFに関わらず、蓄積は個人/パーティの両方に行う
    this._LineageMasterySavevalues[index] = Math.min(this._LineageMasterySavevalues[index] + 1 ,LineageMasteryMaxCount);
    $gameParty.setLineageMasterySavevalues(index);
};

// 系統使い込み値の取得 フラグ次第でパーティ情報から読み出す
Game_Actor.prototype.getLineageMasterySavevalues = function(index) {
    if(PersonalCount == 1){
        if(!this._LineageMasterySavevalues) this.clearLineageMasterySavevalues();
        if (this._LineageMasterySavevalues[index] === undefined){
            this._LineageMasterySavevalues[index] = 0;
        }
        return this._LineageMasterySavevalues[index];
    }
    else{
        return $gameParty.getLineageMasterySavevalues(index);
    }
};

// パーティの系統使い込み値出し入れ

// 系統使い込み値の初期化・クリア
Game_Party.prototype.clearLineageMasterySavevalues = function() {
    this._LineageMasterySavevalues = [];
};

// 系統使い込み値の蓄積
Game_Party.prototype.setLineageMasterySavevalues = function(index) {
    if(!this._LineageMasterySavevalues) this.clearLineageMasterySavevalues();
    if (this._LineageMasterySavevalues[index] === undefined){
        this._LineageMasterySavevalues[index] = 0;
    }
    this._LineageMasterySavevalues[index] = Math.min(this._LineageMasterySavevalues[index] + 1 ,LineageMasteryMaxCount);
};

// 系統使い込み値の取得
Game_Party.prototype.getLineageMasterySavevalues = function(index) {
    if(!this._LineageMasterySavevalues) this.clearLineageMasterySavevalues();
    if (this._LineageMasterySavevalues[index] === undefined){
        this._LineageMasterySavevalues[index] = 0;
    }
    return this._LineageMasterySavevalues[index];
};

// 使い込み値蓄積関数（個別・系統まとめて）
Game_Actor.prototype.setMasterySavevalues = function(item) {
    
    var lineageId = 0;
    if (item && item.meta && item.meta.MasteryLineage) lineageId = Number(item.meta.MasteryLineage);
    
    if (DataManager.isSkill(item)) {
        if(item.id > 0) this.setSkillMasterySavevalues(item.id);
        if(lineageId > 0) this.setLineageMasterySavevalues(lineageId);
    }
    else if (DataManager.isItem(item)) {
        if(lineageId > 0) this.setLineageMasterySavevalues(lineageId);
    }
    
};

const _Game_Battler_useItem = Game_Battler.prototype.useItem;
Game_Battler.prototype.useItem = function(item) {
    _Game_Battler_useItem.call(this, item);
    if(this.isActor()){
        this.setMasterySavevalues(item);
    }
};

// 使い込み値適用関数（個別・系統まとめて）
Game_Actor.prototype.applyMasterySavevalues = function(value,item) {
    
    var skillPlus = 0;
    var lineagePlus = 0;
    var skillMagni = 0;
    var lineageMagni = 0;
    
    var lineageId = 0;
    if (item && item.meta && item.meta.MasteryLineage) lineageId = Number(item.meta.MasteryLineage);

    if (DataManager.isSkill(item)) {
        skillMagni = this.getSkillMasterySavevalues(item.id);
        skillPlus = value * skillMagni / SkillMasteryMaxDamageAmplify;
    }
    if (lineageId > 0) {
        lineageMagni = this.getLineageMasterySavevalues(lineageId);
        lineagePlus = value * lineageMagni / LineageMasteryMaxDamageAmplify;
    }
    
    return Math.round(value + skillPlus + lineagePlus);
};

/* ダメージ計算関数の末尾で使い込み値補正を加味 */
const _Game_Action_makeDamageValue_mastery = Game_Action.prototype.makeDamageValue;
Game_Action.prototype.makeDamageValue = function(target, critical){
    var value = _Game_Action_makeDamageValue_mastery.apply(this, arguments);
    
    if(this.subject().isActor()){
        value = this.subject().applyMasterySavevalues(value,this.item());
    }
    
    return value;
};


// TEST用関数
/*
Game_Party.prototype.setTestDataSkillMastery = function() {
    
    var actor = $gameParty.members().find(actor => actor.actorId() === 2);
    
    actor.clearSkillMasterySavevalues();
    actor.clearLineageMasterySavevalues();
    actor._SkillMasterySavevalues[11] = 500;
    actor._LineageMasterySavevalues[1] = 500;

    var actor = $gameParty.members().find(actor => actor.actorId() === 1);
    actor.clearSkillMasterySavevalues();
    actor.clearLineageMasterySavevalues();
    actor._SkillMasterySavevalues[31] = 250;
    actor._LineageMasterySavevalues[2] = 250;
    
    this.clearSkillMasterySavevalues();
    this.clearLineageMasterySavevalues();
    this._SkillMasterySavevalues[11] = 500;
    this._LineageMasterySavevalues[1] = 500;
    this._SkillMasterySavevalues[31] = 250;
    this._LineageMasterySavevalues[2] = 250;
    
}
*/

})();

